﻿Public Class UniformStack
    Inherits Panel
    Public Shared ReadOnly OrientationProperty =
        DependencyProperty.Register("Orientation",
                                    GetType(Orientation),
                                    GetType(UniformStack),
                                    New PropertyMetadata(Orientation.Vertical, AddressOf OnOrientationChanged))

    Public Property Orientation() As Orientation
        Set(ByVal value As Orientation)
            SetValue(OrientationProperty, value)
        End Set
        Get
            Return CType(GetValue(OrientationProperty), Orientation)
        End Get
    End Property


    Private Shared Sub OnOrientationChanged(ByVal obj As DependencyObject,
                                            ByVal args As DependencyPropertyChangedEventArgs)
        TryCast(obj, UniformStack).InvalidateMeasure()
    End Sub


    Protected Overrides Function MeasureOverride(ByVal availableSize As Size) As Size
        If Children.Count = 0 Then
            Return New Size()
        End If

        Dim availableChildSize As New Size()
        Dim maxChildSize As New Size()
        Dim compositeSize As New Size()

        ' Calculate an available size for each child                
        If Orientation = Orientation.Horizontal Then
            availableChildSize = New Size(availableSize.Width / Children.Count,
                                          availableSize.Height)
        Else
            availableChildSize = New Size(availableSize.Width,
                                          availableSize.Height / Children.Count)
        End If

        ' Enumerate the children, and find the widest width and the highest height
        For Each child In Children
            child.Measure(availableChildSize)
            maxChildSize.Width = Math.Max(maxChildSize.Width, child.DesiredSize.Width)
            maxChildSize.Height = Math.Max(maxChildSize.Height, child.DesiredSize.Height)
        Next child

        ' Now determine a composite size that depends on infinite available width or height
        If Orientation = Orientation.Horizontal Then
            If Double.IsPositiveInfinity(availableSize.Width) Then
                compositeSize = New Size(maxChildSize.Width * Children.Count, maxChildSize.Height)
            Else
                compositeSize = New Size(availableSize.Width, maxChildSize.Height)
            End If
        Else
            If Double.IsPositiveInfinity(availableSize.Height) Then
                compositeSize = New Size(maxChildSize.Width, maxChildSize.Height * Children.Count)
            Else
                compositeSize = New Size(maxChildSize.Width,
                                         availableSize.Height)
            End If
        End If

        Return compositeSize
    End Function


    Protected Overrides Function ArrangeOverride(ByVal finalSize As Size) As Size
        If Children.Count > 0 Then
            Dim finalChildSize As New Size()
            Dim x = 0.0
            Dim y = 0.0

            If Orientation = Orientation.Horizontal Then
                finalChildSize = New Size(finalSize.Width / Children.Count, finalSize.Height)
            Else
                finalChildSize = New Size(finalSize.Width, finalSize.Height / Children.Count)
            End If

            For Each child In Children
                child.Arrange(New Rect(New Point(x, y), finalChildSize))

                If Orientation = Orientation.Horizontal Then
                    x += finalChildSize.Width
                Else
                    y += finalChildSize.Height
                End If
            Next child
        End If

        Return MyBase.ArrangeOverride(finalSize)
    End Function
End Class

