﻿<TemplatePartAttribute(Name:="PlanePart", Type:=GetType(Canvas)), TemplatePartAttribute(Name:="ThumbPart", Type:=GetType(Thumb))>
Public Class XYSlider
    Inherits Control
    Private planePart As Canvas
    Private thumbPart As Thumb
    Private absoluteThumbPoint As Point

    Public Event ValueChanged As RoutedPropertyChangedEventHandler(Of Point)

    Public Shared ReadOnly PlaneBackgroundProperty =
        DependencyProperty.Register("PlaneBackground",
                                    GetType(Brush),
                                    GetType(XYSlider),
                                    New PropertyMetadata(New SolidColorBrush(Colors.Gray)))

    Public Shared ReadOnly ValueProperty =
        DependencyProperty.Register("Value",
                                    GetType(Point),
                                    GetType(XYSlider),
                                    New PropertyMetadata(New Point(0.5, 0.5), AddressOf OnValueChanged))


    Public Sub New()
        Me.DefaultStyleKey = GetType(XYSlider)
    End Sub


    Public Property PlaneBackground() As Brush
        Set(ByVal value As Brush)
            SetValue(PlaneBackgroundProperty, value)
        End Set
        Get
            Return CType(GetValue(PlaneBackgroundProperty), Brush)
        End Get
    End Property


    Public Property Value() As Point
        Set(ByVal value As Point)
            SetValue(ValueProperty, value)
        End Set
        Get
            Return CType(GetValue(ValueProperty), Point)
        End Get
    End Property


    Private Shared Sub OnValueChanged(ByVal obj As DependencyObject,
                                      ByVal args As DependencyPropertyChangedEventArgs)
        TryCast(obj, XYSlider).OnValueChanged(CType(args.OldValue, Point),
                                              CType(args.NewValue, Point))
    End Sub


    Protected Overridable Sub OnValueChanged(ByVal oldValue As Point,
                                             ByVal newValue As Point)
        If newValue.X < 0 OrElse newValue.X > 1 OrElse newValue.Y < 0 OrElse newValue.Y > 1 Then
            Throw New ArgumentOutOfRangeException("Value", "Value property must be Point with coordinates between 0 and 1")
        End If

        ScaleValueToPlane(newValue)

        RaiseEvent ValueChanged(Me,
                                New RoutedPropertyChangedEventArgs(Of Point)(oldValue, newValue))
    End Sub


    Public Overrides Sub OnApplyTemplate()
        If planePart IsNot Nothing Then
            RemoveHandler planePart.SizeChanged, AddressOf OnPlaneSizeChanged
        End If

        If thumbPart IsNot Nothing Then
            RemoveHandler thumbPart.DragDelta, AddressOf OnThumbDragDelta
        End If

        planePart = TryCast(GetTemplateChild("PlanePart"), Canvas)
        thumbPart = TryCast(GetTemplateChild("ThumbPart"), Thumb)

        If planePart IsNot Nothing AndAlso thumbPart IsNot Nothing Then
            AddHandler planePart.SizeChanged, AddressOf OnPlaneSizeChanged
            AddHandler thumbPart.DragStarted, AddressOf OnThumbDragStarted
            AddHandler thumbPart.DragDelta, AddressOf OnThumbDragDelta
            ScaleValueToPlane(Me.Value)
        End If

        MyBase.OnApplyTemplate()
    End Sub


    Private Sub OnPlaneSizeChanged(ByVal sender As Object, ByVal args As SizeChangedEventArgs)
        ScaleValueToPlane(Me.Value)
    End Sub


    Private Sub OnThumbDragStarted(ByVal sender As Object, ByVal args As DragStartedEventArgs)
        absoluteThumbPoint = New Point(Canvas.GetLeft(thumbPart), Canvas.GetTop(thumbPart))
    End Sub


    Private Sub OnThumbDragDelta(ByVal sender As Object, ByVal args As DragDeltaEventArgs)
        absoluteThumbPoint.X += args.HorizontalChange
        absoluteThumbPoint.Y += args.VerticalChange

        Value = New Point(Math.Max(0, Math.Min(1, absoluteThumbPoint.X / planePart.ActualWidth)),
                          Math.Max(0, Math.Min(1, absoluteThumbPoint.Y / planePart.ActualHeight)))
    End Sub


    Private Sub ScaleValueToPlane(ByVal point As Point)
        If planePart IsNot Nothing AndAlso thumbPart IsNot Nothing Then
            Canvas.SetLeft(thumbPart, planePart.ActualWidth * point.X)
            Canvas.SetTop(thumbPart, planePart.ActualHeight * point.Y)
        End If
    End Sub
End Class

