﻿Public Class OverlapPanel
    Inherits Panel
    Private maxChildSize As New Size()

    Public Shared ReadOnly OrientationProperty =
        DependencyProperty.Register("Orientation",
                                    GetType(Orientation),
                                    GetType(OverlapPanel),
                                    New PropertyMetadata(Orientation.Horizontal, AddressOf OnAffectsMeasure))


    Public Shared ReadOnly MinimumOverlapProperty =
        DependencyProperty.Register("MinimumOverlap",
                                    GetType(Double),
                                    GetType(OverlapPanel),
                                    New PropertyMetadata(0.0, AddressOf OnAffectsMeasure))

    Public Property Orientation() As Orientation
        Set(ByVal value As Orientation)
            SetValue(OrientationProperty, value)
        End Set
        Get
            Return CType(GetValue(OrientationProperty), Orientation)
        End Get
    End Property


    Public Property MinimumOverlap() As Double
        Set(ByVal value As Double)
            SetValue(MinimumOverlapProperty, value)
        End Set
        Get
            Return CDbl(GetValue(MinimumOverlapProperty))
        End Get
    End Property


    Private Shared Sub OnAffectsMeasure(ByVal obj As DependencyObject, ByVal args As DependencyPropertyChangedEventArgs)
        TryCast(obj, OverlapPanel).InvalidateMeasure()
    End Sub


    Protected Overrides Function MeasureOverride(ByVal availableSize As Size) As Size
        If Children.Count = 0 Then
            Return New Size(0, 0)
        End If

        maxChildSize = New Size()

        For Each child In Children
            If Orientation = Orientation.Horizontal Then
                child.Measure(New Size(Double.PositiveInfinity, availableSize.Height))
            Else
                child.Measure(New Size(availableSize.Width, Double.PositiveInfinity))
            End If

            maxChildSize.Width = Math.Max(maxChildSize.Width, child.DesiredSize.Width)

            maxChildSize.Height = Math.Max(maxChildSize.Height, child.DesiredSize.Height)
        Next child

        If Orientation = Orientation.Horizontal Then
            Dim maxTotalWidth = maxChildSize.Width * Children.Count
            Dim minTotalWidth = maxChildSize.Width + MinimumOverlap * (Children.Count - 1)

            If Double.IsPositiveInfinity(availableSize.Width) Then
                Return New Size(minTotalWidth, maxChildSize.Height)
            End If

            If maxTotalWidth < availableSize.Width Then
                Return New Size(maxTotalWidth, maxChildSize.Height)

            ElseIf minTotalWidth < availableSize.Width Then
                Return New Size(availableSize.Width, maxChildSize.Height)
            End If

            Return New Size(minTotalWidth, maxChildSize.Height)
        End If
        ' Orientation = Vertical

        Dim maxTotalHeight = maxChildSize.Height * Children.Count
        Dim minTotalHeight = maxChildSize.Height + MinimumOverlap * (Children.Count - 1)

        If Double.IsPositiveInfinity(availableSize.Height) Then
            Return New Size(maxChildSize.Width, minTotalHeight)
        End If

        If maxTotalHeight < availableSize.Height Then
            Return New Size(maxChildSize.Width, maxTotalHeight)

        ElseIf minTotalHeight < availableSize.Height Then
            Return New Size(maxChildSize.Width, availableSize.Height)
        End If

        Return New Size(maxChildSize.Width, minTotalHeight)
    End Function


    Protected Overrides Function ArrangeOverride(ByVal finalSize As Size) As Size
        If Children.Count = 0 Then
            Return finalSize
        End If

        Dim increment = 0.0

        If Orientation = Orientation.Horizontal Then
            increment = Math.Max(MinimumOverlap,
                                 (finalSize.Width - maxChildSize.Width) / (Children.Count - 1))
        Else
            increment = Math.Max(MinimumOverlap,
                                 (finalSize.Height - maxChildSize.Height) / (Children.Count - 1))
        End If

        Dim ptChild As New Point()

        For Each child In Children
            child.Arrange(New Rect(ptChild, maxChildSize))

            If Orientation = Orientation.Horizontal Then
                ptChild.X += increment
            Else
                ptChild.Y += increment
            End If
        Next child

        Return finalSize
    End Function
End Class

